#include <iostream>
#include <getopt.h>

#include "mcat.hpp"

using namespace mcat;

/**
 * Print the usage
 */
void usage(std::string prg_name)
{
    std::cerr << "Usage: " << prg_name << " [options] input_mrl [output_mrls]" << std::endl;
    std::cerr << "       A MRL is built as \"protocol://[hostname[:port]][path]\"" << std::endl;
    std::cerr << "       The protocol `file` may be omited." << std::endl;
    std::cerr << "       For TCP and UDP, the program supports both IPv4 and IPv6." << std::endl;
    std::cerr << "       The ouput MRLs are separated by spaces." << std::endl;
    std::cerr << "Options:"  << std::endl;
    std::cerr << "  -c --buffer-count <count>   set the number of buffers (5 by default)" << std::endl;
    std::cerr << "  -s --buffer-size <size>     set the size of each buffer (1500 by default)" << std::endl;
    std::cerr << "  -b --broadcast              enable broadcast on the interfaces" << std::endl;
    std::cerr << "  -h --help                   print this message and exit" << std::endl;
    Mcat::printUsage(std::cerr);
}

/**
 * Main entry point of the application
 * @param argc: the number of argument
 * @param argv; the arguments
 * @return 1 in case of error and 0 overwise
 */
int main(int argc, char *argv[])
{
    // The input mrl and the vector of output mrls
    std::vector<std::string> outputs;
    std::string input;

    // The size and number of buffer than might come from configuration
    unsigned int buffer_count = 5;
    size_t       buffer_size = 1500;
    bool         broadcast = false;

    // Use getopt to get the arguments
    static const option long_options[] =
    {
        {"buffer-count",    required_argument, NULL, 'c'},
        {"buffer-size",     required_argument, NULL, 's'},
        {"broadcast",       no_argument,       NULL, 'b'},
        {"help",            no_argument,       NULL, 'h'},
        {0, 0, 0, 0}
    };

    while(true)
    {
        int option_index, c;

        if((c = getopt_long(argc, argv, "c:s:hb", long_options, &option_index)) == -1)
            break;

        // Find the associated options
        switch(c)
        {
        case 'c':
            buffer_count = atoi(optarg);
            break;
        case 's':
            buffer_size = atoi(optarg);
            break;
        case 'b':
            broadcast = true;
            break;
        case 'h':
        case '?':
        default:
            usage(argv[0]);
            return c == 'h' ? 0 : 1;
        }
    }

    // Check that we have enought arguments now
    if(argc <= optind)
    {
        usage(argv[0]);
        return 1;
    }

    // The input mrl
    input = argv[optind];
    optind++;

    // If there is only one mrl, the output is the std://
    if(argc == optind)
        outputs.push_back("std://");
    else
    {
        // Build the vector of ouputs
        outputs.reserve(argc - optind);
        for(int i = optind; i < argc; i++)
            outputs.push_back(argv[i]);
    }

    // Launch multi-cat
    Mcat mcat_main(input, outputs, buffer_count, buffer_size, broadcast);

    return 0;
}
