#include <cstdlib>
#include <cstring>
#include <iostream>

#include <netdb.h>
#include <sys/socket.h>
#include <arpa/inet.h>

#include "common.hpp"
#include "macros.hpp"

using namespace std;
using namespace mcat;

typedef struct
{
    const char *mrl;
    const char *ip;
    bool ipv6;
}sample_t;

static const sample_t samples[] =
{
    { "tcp://localhost",        "127.0.0.1",                        false },
    { "tcp://www.dinauz.org",   "91.121.7.193",                     false },
    { "tcp://www.ecp.fr",       "138.195.9.16",                     false },
    { "tcp://www.via.ecp.fr",   "138.195.130.53",                   false },
    { "tcp://134.1.2.3",        "134.1.2.3",                        false },
// not defined on every computer
//    { "tcp://localhost",        "::1",                              true  },
    { "tcp://www.ipv6.org",     "2001:6b0:1:ea:202:a5ff:fecd:13a6", true  },
    { "tcp://www.via.ecp.fr",   "2002:8ac3:802d:1243::53",          true  },
};
static const size_t samples_count = sizeof(samples)/sizeof(sample_t);


const char *inet6_ntoa (struct in6_addr *addr)
{
    static char buf[INET6_ADDRSTRLEN];

    inet_ntop (AF_INET6, addr, buf, INET6_ADDRSTRLEN);
    return buf;
}


int test_resolv(Mrl mrl, const char *ip, bool ipv6)
{
    addrinfo *p_addr_info;
    if(!getAddressInfo(mrl, ipv6 ? AF_INET6 : AF_INET, SOCK_STREAM, &p_addr_info))
    {
        cout << endl << "Failed to resolv the dns " << mrl.getHostname() << endl;
        return 1;
    }

    if(ipv6)
    {
        in6_addr *in6addr = &((struct sockaddr_in6*)p_addr_info->ai_addr)->sin6_addr;
        if(strcmp(inet6_ntoa(in6addr), ip))
        {
            cout << endl << "Wrong ip " << inet6_ntoa(in6addr) << " instead of " << ip << endl;
            freeaddrinfo(p_addr_info);
            return 1;
        }
    }
    else
    {
        in_addr inaddr = ((struct sockaddr_in*)p_addr_info->ai_addr)->sin_addr;
        if(strcmp(inet_ntoa(inaddr), ip))
        {
            cout << endl << "Wrong ip " << inet_ntoa(inaddr) << " instead of " << ip << endl;
            freeaddrinfo(p_addr_info);
            return 1;
        }
    }
    freeaddrinfo(p_addr_info);
    return 0;
}

int main()
{
    cout << "Test getAddressInfo function...";
    cout.flush();

    for(size_t i = 0; i < samples_count; i++)
        if(test_resolv(Mrl(samples[i].mrl), samples[i].ip, samples[i].ipv6))
            return 1;

    cout << "\t[done]" << endl;
}
