#include <cstdlib>
#include <iostream>
#include <cstring>

#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netdb.h>

#include "common.hpp"
#include "macros.hpp"
#include "input_udp.hpp"

using namespace mcat;

InputUDP::InputUDP(Mrl &mrl, bool broadcast) : Input(mrl)
{
    if(!getAddressInfo(mrl, AF_UNSPEC, SOCK_DGRAM, &p_addr_info))
        return;

    i_sock = socket(p_addr_info->ai_family == AF_INET ? PF_INET : PF_INET6,
                    SOCK_DGRAM, IPPROTO_UDP);
    if (i_sock < 0)
    {
        std::cerr << "Socket creation failed" << std::endl;
        return;
    }

    if (broadcast) enableBroadcast(i_sock);

    /* inet4/6 specific part */
    if (p_addr_info->ai_family == AF_INET)
    {
        uint32_t s_addr4 = ((struct sockaddr_in*)p_addr_info->ai_addr)->sin_addr.s_addr;
        if (IN_MULTICAST(ntohl(s_addr4)))
        {
            int flag_true = 1;
            /* join the multicast group */
#ifdef __linux__
            struct ip_mreqn imr;
            imr.imr_multiaddr.s_addr = s_addr4;
            imr.imr_address.s_addr = INADDR_ANY;
            imr.imr_ifindex = 0;
#else
            struct ip_mreq imr;
            imr.imr_multiaddr.s_addr = s_addr4;
            imr.imr_interface.s_addr = INADDR_ANY;
#endif

            if ((setsockopt(i_sock, SOL_SOCKET, SO_REUSEADDR,
                            (void*)&flag_true, sizeof(flag_true))) < 0 ||
                (setsockopt(i_sock, IPPROTO_IP, IP_ADD_MEMBERSHIP,
                            (void*)&imr, sizeof(imr))) < 0)
            {
                std::cerr << "setsockopt() failed" << std::endl;
                return;
            }
        }
    }
    else
    {
        struct in6_addr* addr_6 = &((struct sockaddr_in6*)p_addr_info->ai_addr)->sin6_addr;
        if (IN6_IS_ADDR_MULTICAST(addr_6))
        {
            int flag_true = 1;
            /* join the multicast group */
            struct ipv6_mreq imr;
            memcpy(&imr.ipv6mr_multiaddr, p_addr_info->ai_addr, p_addr_info->ai_addrlen);
            imr.ipv6mr_interface = 0;

            if ((setsockopt(i_sock, SOL_SOCKET, SO_REUSEADDR,
                            (void*)&flag_true, sizeof(flag_true))) < 0 ||
                (setsockopt(i_sock, IPPROTO_IPV6, IPV6_ADD_MEMBERSHIP,
                            (void*)&imr, sizeof(imr))) < 0)
            {
                std::cerr << "setsockopt() failed" << std::endl;
                return;
            }
        }
    }

    if (bind(i_sock, (struct sockaddr*)p_addr_info->ai_addr, p_addr_info->ai_addrlen) < 0) 
    {
        std::cerr << "Could not bind to address" << std::endl;
        return;
    }

    // The input is now ready
    is_ready = true;
}

InputUDP::~InputUDP()
{
    /* membership to multicast group automatically cancelled */
    close(i_sock);
    freeaddrinfo(p_addr_info);
}

int InputUDP::read(char* p_buffer, ssize_t ui_size)
{
    socklen_t len = p_addr_info->ai_addrlen;
    /* if len > sizeof(s_addr) the result has been cut down => check it ? */
    return recvfrom(i_sock, p_buffer, ui_size, 0,
                  (struct sockaddr *) p_addr_info->ai_addr, &len);
}

