#include <iostream>
#include <cstdlib>
#include <cstring>

#include <sys/types.h>
#include <arpa/inet.h>
#include <sys/socket.h>
#include <netdb.h>

#include "common.hpp"
#include "macros.hpp"

#include "output_udp.hpp"

#define DEFAULT_TTL 1

using namespace mcat;

OutputUDP::OutputUDP(Mrl &mrl, bool broadcast) : Output(mrl), ttl(DEFAULT_TTL)
{
    if(!getAddressInfo(mrl, AF_UNSPEC, SOCK_DGRAM, &p_addr_info))
        return;

    i_sock = socket(p_addr_info->ai_family == AF_INET ? PF_INET : PF_INET6,
                    SOCK_DGRAM, IPPROTO_UDP);
    if (i_sock < 0)
    {
        std::cerr << "Socket creation failed" << std::endl;
        return;
    }

    if (broadcast) enableBroadcast(i_sock);

    /* inet4/6 specific part */
    if (p_addr_info->ai_family == AF_INET)
    {
        uint32_t s_addr4 = ((struct sockaddr_in*)p_addr_info->ai_addr)->sin_addr.s_addr;
        if (IN_MULTICAST(ntohl(s_addr4)))
        {
            unsigned char flag_true = 1;
            if ((setsockopt(i_sock, IPPROTO_IP, IP_MULTICAST_TTL,
                            (void*) &ttl, sizeof(ttl))) < 0 ||
                (setsockopt(i_sock, IPPROTO_IP, IP_MULTICAST_LOOP,
                            (void*) &flag_true, sizeof(flag_true))) < 0)
            {
                std::cerr << "setsockopt() failed" << std::endl;
                return;
            }
        }
    }
    else
    {
        struct in6_addr* addr_6 = &((struct sockaddr_in6*)p_addr_info->ai_addr)->sin6_addr;
        if (IN6_IS_ADDR_MULTICAST(addr_6))
        {
            unsigned char flag_true = 1;
            if ((setsockopt(i_sock, IPPROTO_IPV6, IPV6_MULTICAST_HOPS,
                            (void*) &ttl, sizeof(ttl))) < 0 ||
                (setsockopt(i_sock, IPPROTO_IPV6, IPV6_MULTICAST_LOOP,
                            (void*) &flag_true, sizeof(flag_true))) < 0)
            {
                std::cerr << "setsockopt() failed" << std::endl;
                return;
            }
        }
    }

    // The output is now ready
    is_ready = true;
}

OutputUDP::~OutputUDP()
{
    close(i_sock);
    freeaddrinfo(p_addr_info);
}

int OutputUDP::write(const char *p_buffer, ssize_t ui_size)
{
    return sendto(i_sock, p_buffer, ui_size, 0,
                  (struct sockaddr *) p_addr_info->ai_addr,
                  p_addr_info->ai_addrlen);
}
