#include <iostream>

#include "mcat.hpp"

#include "input.hpp"
#include "input_file.hpp"
#include "input_std.hpp"
#include "input_tcp.hpp"
#include "input_udp.hpp"

#include "output.hpp"
#include "output_file.hpp"
#include "output_std.hpp"
#include "output_udp.hpp"
#include "output_tcp.hpp"

using namespace mcat;

Mcat::Mcat(std::string input, std::vector<std::string> outputs,
           unsigned int buffer_count, size_t buffer_size,
           bool broadcast)
{
    // Create the ring buffer
    m_buffers = new RingBuffer(buffer_size, buffer_count);
    m_buffer_size = buffer_size;

    // Create the input
    Mrl input_mrl(input);
    std::string protocol = input_mrl.getProtocol();
    if(!protocol.compare(InputFile::getProtocol()))
        m_input = new InputFile(input_mrl);
    else if(!protocol.compare(InputStd::getProtocol()))
        m_input = new InputStd(input_mrl);
    else if(!protocol.compare(InputTCP::getProtocol()))
        m_input = new InputTCP(input_mrl);
    else if(!protocol.compare(InputUDP::getProtocol()))
        m_input = new InputUDP(input_mrl, broadcast);
    else
    {
        std::cerr << "Unimplemeted protocol for the input: " << protocol << std::endl;
        return;
    }

    // Check that the input was created corretly
    if(!m_input->ready())
        return;

    // Create the outputs
    m_output.resize(outputs.size());
    std::vector<std::string>::iterator vit;
    size_t index = 0;
    for(vit = outputs.begin(); vit != outputs.end(); vit++, index++)
    {
        Mrl output_mrl(*vit);
        std::string out_proto = output_mrl.getProtocol();
        if(!out_proto.compare(OutputFile::getProtocol()))
            m_output[index] = new OutputFile(output_mrl);
        else if(!out_proto.compare(OutputStd::getProtocol()))
            m_output[index] = new OutputStd(output_mrl);
        else if(!out_proto.compare(OutputUDP::getProtocol()))
            m_output[index] = new OutputUDP(output_mrl, broadcast);
        else if(!out_proto.compare(OutputTCP::getProtocol()))
            m_output[index] = new OutputTCP(output_mrl);
        else
        {
            std::cerr << "Unimplemeted protocole for the output: " << out_proto << std::endl;
            return;
        }

        // Check that the input is ready
        if(!m_output[index]->ready())
            return;
    }

    // The input and the outputs are ready: launch the threads
    if(pthread_create(&m_input_thread, NULL, launch_input_thread, (void*)this))
    {
        std::cerr << "Unable to launch the input thread" << std::endl;
        return;
    }

    if(pthread_create(&m_output_thread, NULL, launch_output_thread, (void*)this))
    {
        std::cerr << "Unable to launch the output thread" << std::endl;
        return;
    }

    // Wait for the input and output thread
    pthread_join(m_input_thread, NULL);
    pthread_join(m_output_thread, NULL);
}

Mcat::~Mcat()
{
    std::vector<Output*>::iterator vit;
    for(vit = m_output.begin(); vit != m_output.end(); vit++)
        delete (*vit);

    delete m_input;
    delete m_buffers;
}

void Mcat::printUsage(std::ostream& stream)
{
    stream << "Inputs:" << std::endl;
    stream << "  - " << InputFile::getProtocol() << "    " <<  InputFile::getUsage() << std::endl;
    stream << "  - " << InputStd::getProtocol() << "     " <<  InputStd::getUsage()  << std::endl;
    stream << "  - " << InputTCP::getProtocol() << "     " <<  InputTCP::getUsage()  << std::endl;
    stream << "  - " << InputUDP::getProtocol() << "     " <<  InputUDP::getUsage()  << std::endl;

    stream << "Outputs:" << std::endl;
    stream << "  - " << OutputFile::getProtocol() << "    " <<  OutputFile::getUsage() << std::endl;
    stream << "  - " << OutputStd::getProtocol() << "     " <<  OutputStd::getUsage()  << std::endl;
    stream << "  - " << OutputTCP::getProtocol() << "     " <<  OutputTCP::getUsage()  << std::endl;
    stream << "  - " << OutputUDP::getProtocol() << "     " <<  OutputUDP::getUsage()  << std::endl;
}

void *Mcat::launch_input_thread(void *p_data)
{
    Mcat *p_this = static_cast<Mcat*>(p_data);

    p_this->input_thread();
    return NULL;
}

void Mcat::input_thread()
{
    bool loop;
    do
    {
        Buffer *p_buffer = m_buffers->nextEmpty();
        loop = (p_buffer->i_size = m_input->read(p_buffer->p_buffer, m_buffer_size)) > 0;
        m_buffers->pushFull(p_buffer);
    }
    while(loop);
}

void *Mcat::launch_output_thread(void *p_data)
{
    Mcat *p_this = static_cast<Mcat*>(p_data);

    p_this->output_thread();
    return NULL;
}

void Mcat::output_thread()
{
    bool loop;
    do
    {
        const Buffer *p_buffer = m_buffers->nextFull();
        loop = p_buffer->i_size > 0;

        std::vector<Output*>::iterator vit;
        for(vit = m_output.begin(); vit != m_output.end(); vit++)
            if ((*vit)->write(p_buffer->p_buffer, p_buffer->i_size) < 0)
                std::cerr << "error while writing to an output" << std::endl;
        m_buffers->popFull();
    }
    while(loop);
}

