#!/bin/sh

if [ -z "$1" ] ; then
    echo "Usage: $0 SYSTEM [GCC [\"CFLAGS\"]]"
    echo "   SYSTEM: UINT32 | UINT64 | MMX | SSE | ALTIVEC"
    exit 1
fi

cd `dirname $0`

if [ ! -d "build" ] ; then
    mkdir build
fi
cd build

SYSTEM="$1"
ARG_GCC="$2"
ARG_CFLAGS="$3"

VER="1.1.0"
ARC="libdvbcsa-$VER.tar.gz"
URL="http://cesbo.com/download/astra/contrib/$ARC"

MAKEFILE="Makefile"
CONFIG="src/config.h"

err()
{
    CDIR=`pwd`
    echo "failed to download libdvbcsa"
    echo "please, download $URL"
    echo "to $CDIR"
    exit 1
}

download()
{
    if [ -d "libdvbcsa" ] ; then
        return 0
    fi

    if [ ! -f "$ARC" ] ; then
        DCMD=""

        if which curl >/dev/null ; then
            DCMD="curl -O"
        elif which wget >/dev/null ; then
            DCMD="wget"
        elif which fetch >/dev/null ; then
            DCMD="fetch"
        else
            err
        fi

        $DCMD $URL
        if [ $? -ne 0 ] ; then
            err
        fi
    fi

    tar -xf $ARC
    mv libdvbcsa-$VER libdvbcsa
}

download
cd libdvbcsa

GCC="gcc"
if [ -n "$ARG_GCC" ] ; then
    GCC="$ARG_GCC"
fi
AR=`echo $GCC | sed 's/gcc$/ar/'`

CFLAGS="-O3 -fomit-frame-pointer -funroll-loops"
if [ -n "$ARG_CFLAGS" ] ; then
    CFLAGS="$CFLAGS $ARG_CFLAGS"
fi

case "$SYSTEM" in
"UINT32")
    TRANSPOSE="32"
    ;;
"UINT64")
    TRANSPOSE="64"
    ;;
"MMX")
    TRANSPOSE="64"
    CFLAGS="$CFLAGS -mmmx"
    ;;
"SSE")
    TRANSPOSE="128"
    CFLAGS="$CFLAGS -msse -msse2"
    ;;
"ALTIVEC")
    TRANSPOSE="128"
    ;;
*)
    echo "wrong SYSTEM option"
    exit 1
    ;;
esac

cat >$CONFIG <<EOF
#define STDC_HEADERS 1

#define DVBCSA_USE_$SYSTEM 1

#define HAVE_STDLIB_H 1
#define HAVE_STRING_H 1
#define HAVE_STDINT_H 1

EOF

posix_memalign_test_c()
{
    cat <<EOF
#include <stdio.h>
#include <stdlib.h>
int main(void) { void *p = NULL; return posix_memalign(&p, 32, 128); }
EOF
}

check_posix_memalign()
{
    posix_memalign_test_c | $GCC -Werror -o .link-test -x c - >/dev/null 2>&1
    if [ $? -eq 0 ] ; then
        rm -f .link-test
        return 0
    else
        return 1
    fi
}

if check_posix_memalign ; then
    echo "#define HAVE_POSIX_MEMALIGN 1" >>$CONFIG
fi

mm_malloc_test_c()
{
    cat <<EOF
#include <mm_malloc.h>
int main(void) { void *p = _mm_malloc(8,8); _mm_free(p); return 0; }
EOF
}

check_mm_malloc()
{
    mm_malloc_test_c | $GCC -Werror -o .link-test -x c - >/dev/null 2>&1
    if [ $? -eq 0 ] ; then
        rm -f .link-test
        return 0
    else
        return 1
    fi
}

if check_mm_malloc ; then
    echo "#define HAVE_MM_MALLOC 1" >>$CONFIG
fi

OUT=""
LNK=""

MACHINE=`$GCC -dumpmachine`
case "$MACHINE" in
*"darwin"*)
    OUT="libdvbcsa.$VER.dylib"
    LNK="libdvbcsa.dylib"
    ;;
*"mingw"*)
    OUT="libdvbcsa.dll"
    ;;
*)
    CFLAGS="$CFLAGS -fPIC"
    OUT="libdvbcsa.so.$VER"
    LNK="libdvbcsa.so"
esac

cat >$MAKEFILE <<EOF
# generated by libdvbcsa.sh

MAKEFLAGS = -rR --no-print-directory

OUT = $OUT
LNK = $LNK
ALIB = libdvbcsa.a

LPATH = /usr/lib
IPATH = /usr/include/dvbcsa

CC = $GCC
LD = $GCC
AR = $AR

CFLAGS = $CFLAGS
LDFLAGS = -shared

OBJS = src/dvbcsa_key.o src/dvbcsa_block.o src/dvbcsa_algo.o src/dvbcsa_stream.o
BSOBJS = src/dvbcsa_bs_algo.o src/dvbcsa_bs_block.o src/dvbcsa_bs_key.o src/dvbcsa_bs_stream.o \
  src/dvbcsa_bs_transpose.o src/dvbcsa_bs_transpose$TRANSPOSE.o

.PHONY: all clean install

all: \$(OUT) \$(ALIB)

clean:
	@echo "CLEAN"
	@rm -f \$(OUT) \$(ALIB) \$(OBJS) \$(BSOBJS) $MAKEFILE $CONFIG

\$(OUT): \$(OBJS) \$(BSOBJS)
	@echo "BUILD: \$@"
	@\$(LD) \$(LDFLAGS) \$^ -o \$@

\$(ALIB): \$(OBJS) \$(BSOBJS)
	@echo "BUILD: \$@"
	@\$(AR) cru \$@ \$^

%.o: %.c
	@echo "   CC: \$@"
	@\$(CC) \$(CFLAGS) -c \$< -o \$@

install: \$(OUT)
	@echo "INSTALL: \$(LPATH)/\$(OUT)"
	@cp \$(OUT) \$(LPATH)/\$(OUT)
	@echo "INSTALL: \$(LPATH)/\$(LNK)"
	@ln -nfs \$(OUT) \$(LPATH)/\$(LNK)
	@echo "INSTALL: \$(IPATH)/dvbcsa.h"
	@mkdir -p \$(IPATH)
	@cp src/dvbcsa/dvbcsa.h \$(IPATH)/dvbcsa.h

EOF

make
if [ $? -eq 0 ] ; then
    echo ""
    echo "make complete"
    echo "to install libdvbcsa use: sudo make -C build/libdvbcsa install"
    echo "or build astra for built-in use"
    exit 0
fi

exit 1
